// External variables
const express = require("express");
const bodyParser = require("body-parser");
const mongoose = require("mongoose");
const bcrypt = require("bcryptjs");
const cors = require("cors");
const cookies = require("cookie-parser");
const http = require("http");
const { Server } = require("socket.io");
const fs = require("fs");
const path = require("path");
require("dotenv").config(); // ✅ This must be at the very top before MongoURI

// ✅ Use env variable properly
const MongoURI = process.env.MONGO_URI;
const port = process.env.PORT || 8001;

// Check env loaded
if (!MongoURI) {
  console.error("❌ MONGO_URI not found in .env file!");
  process.exit(1);
}

// Controllers and Routes
const { patientRegister } = require("./Routes/userController");
const { createPharmacistReq, searchMedicine } = require("./Routes/pharmacistController");
const { addAdmin, viewPatientDet, PatientDetailsResults } = require("./Routes/adminController");
const admin = require("./Routers/adminRoute");
const pharmacist = require("./Routers/pharmacistRoute");
const patient = require("./Routers/patientRoute");
const auth = require("./Routers/authRoute");
const cart = require("./Routers/cartRoute");
const order = require("./Routers/orderRoute");

// App setup
const app = express();
app.use(express.static("uploads"));
app.engine("html", require("ejs").renderFile);
app.set("view engine", "html");
app.set("views", path.join(__dirname, "Views"));
app.use(bodyParser.urlencoded({ extended: false }));
app.use(express.json());
app.use(cookies());

// CORS setup
const corsOptions = {
  origin: "http://localhost:3001",
  methods: ["GET", "POST", "PUT", "DELETE"],
  credentials: true,
};
app.use(cors(corsOptions));

// Socket.io setup
const server = http.createServer(app);
const io = new Server(server, {
  cors: {
    origin: "http://localhost:3001",
    methods: ["GET", "POST"],
  },
});

io.on("connection", (socket) => {
  console.log(`✅ User Connected: ${socket.id}`);

  socket.on("join_room", (data) => {
    socket.join(data);
    console.log(`User ${socket.id} joined room: ${data}`);
  });

  socket.on("send_message", (data) => {
    socket.to(data.room).emit("receive_message", data);
  });

  socket.on("disconnect", () => {
    console.log(`❌ User disconnected: ${socket.id}`);
  });
});

// MongoDB connection
mongoose
  .connect(MongoURI, {
    useNewUrlParser: true,
    useUnifiedTopology: true,
  })
  .then(() => {
    console.log("✅ MongoDB is now connected!");
    server.listen(port, () => {
      console.log(`🚀 Listening to requests on http://localhost:${port}`);
    });
  })
  .catch((err) => console.error("❌ MongoDB connection error:", err));

// Routes
app.use("/", auth);
app.use("/pharmacist", pharmacist);
app.use("/admin", admin);
app.use("/patient", patient);
app.use("/cart", cart);
app.use("/order", order);
